'
' MMA7260 Accelerometer interface
'
' Display the X, Y and Z axis g-force values and angles
'
' Accelerometer connections
'
' X output connects to AN0 (PORTA.0)
' Y output connects to AN1 (PORTA.1)
' Z output connects to AN2 (PORTA.2)
' Sel1 input connects to PORTB.5
' Sel2 input connects to PORTB.0
' Sleep connects to VDD
'
    Include "Amicus18.inc"          ' Configure the compiler to use the Amicus18 hardware (18F25K20, 64MHz)
    Include "Amicus18_ADC.inc"      ' Load the Amicus18 ADC macro's into the program

    Symbol tSel1 = PORTB.5          ' Accelerometer Sel1 pin connection
    Symbol tSel2 = PORTB.0          ' Accelerometer Sel2 pin connection

    Symbol cQuanta = 3.3 / 1023     ' ADC to voltage conversion (VDD / ADC resolution)
    Symbol cDegreeConversion = 180.0 / 3.14159265 ' Conversion from radians to degrees

    Dim wXoutput As Word            ' Holds the filtered 10-Bit result from the ADC
    Dim wYoutput As Word            ' Holds the filtered 10-Bit result from the ADC
    Dim wZoutput As Word            ' Holds the filtered 10-Bit result from the ADC

    Dim fXVoltage As Float          ' Holds the voltage output from the channel
    Dim fYVoltage As Float          ' Holds the voltage output from the channel
    Dim fZVoltage As Float          ' Holds the voltage output from the channel

    Dim R As Float                  ' Holds the result of the multiple sqr

    Dim Rx As Float                 ' Holds the result of  fGravity_X squared
    Dim Ry As Float                 ' Holds the result of  fGravity_Y squared
    Dim Rz As Float                 ' Holds the result of  fGravity_Z squared

    Dim bAngleX As Byte              ' Holds the inclination angle of the X axis
    Dim bAngleY As Byte              ' Holds the inclination angle of the Y axis
    
    Dim fGravity_X As Float         ' Holds the result of the X axis g-force calculation
    Dim fGravity_Y As Float         ' Holds the result of the Y axis g-force calculation
    Dim fGravity_Z As Float         ' Holds the result of the Z axis g-force calculation

    Symbol Zero_G_X = 1.57          ' Zero g voltage level for X axis, as measured
    Symbol Zero_G_Y = 1.55          ' Zero g voltage level for Y axis, as measured
    Symbol Zero_G_Z = 1.65          ' Zero g voltage level for Z axis, as measured
'
' Variables used by the bubble sort routine
'
    Dim bSortIndex As Byte
    Dim tSwapOccured As Bit
    Dim wSwapTemp As Word
    Dim wSortResult As wSwapTemp

    Symbol cSamplesToTake = 19       ' The amount of samples to take for the bubble sort
    Dim wSampleArray[cSamplesToTake] As Word

'--------------------------------------------------------------------
    GoTo Main                       ' Jump over the subroutines
'--------------------------------------------------------------------
'
' Select the sensitivity of the MMA7620
' Also creates a sensitivity constant for each g selection
'
$define Set_1_5g()  '
    Symbol cSensitivity = 0.8 '
    Low tSel1       '
    Low tSel2

$define Set_2g()    '
    Symbol cSensitivity = 0.6 '
    High tSel1      '
    Low tSel2

$define Set_4g()    '
    Symbol cSensitivity = 0.3 '
    Low tSel1       '
    High tSel2

$define Set_6g()    '
    Symbol cSensitivity = 0.2 '
    High tSel1      '
    High tSel2
'--------------------------------------------------------------------
' Implement a bubble sorting median filter
'
' Input     : wSampleArray holds the values to sort
' Output    : wSortResult holds the median of the values
' Notes     : None
'
MedianFilter:
    Repeat
        tSwapOccured = 0                                                    ' Clear flag that indicates swap.
        bSortIndex = 0
        Repeat                                                              ' For each cell of the array...
            If wSampleArray[bSortIndex] > wSampleArray[bSortIndex + 1] Then ' Move larger values up.
                wSwapTemp = wSampleArray[bSortIndex]                        ' ..by swapping them.
                wSampleArray[bSortIndex] = wSampleArray[bSortIndex + 1]
                wSampleArray[bSortIndex + 1] = wSwapTemp
                tSwapOccured = 1                                            ' Set bit if swap occurred.
            EndIf
            Inc bSortIndex
        Until bSortIndex > cSamplesToTake                                   ' Check next cell of the array.
    Until tSwapOccured = 0                                                  ' Keep sorting until no more swaps.
    wSortResult = wSampleArray[cSamplesToTake / 2]                          ' Extract the middle (median) value
    Return
'--------------------------------------------------------------------
' Read the X output from the accelerometer (Forward-Backward)
'
' Input     : None
' Output    : wXoutput holds the filtered 10-bit result from the ADC
'             fXVoltage holds the voltage output from the channel
' Notes     : None
ReadChannel_X:
    For bSortIndex = (cSamplesToTake - 1) To 0 Step -1  ' Create a loop for the amount of samples to take
        wSampleArray[bSortIndex] = ReadADC(ADC_CH0)     ' Read the channel into the filter array
        DelayUS 2                                       ' A small delay to allow the ADC's capacitor to re-charge
    Next                                                ' Close the loop
    GoSub MedianFilter                                  ' Perform a median filter on the samples
    wXoutput = wSortResult                              ' Transfer the result of the filter
    fXVoltage = wXoutput * cQuanta                      ' Convert to a voltage
    Return
'--------------------------------------------------------------------
' Read the Y output from the accelerometer (Side to Side)
'
' Input     : None
' Output    : wYoutput holds the filtered 10-bit result from the ADC
'             fYVoltage holds the voltage output from the channel
' Notes     : None
ReadChannel_Y:
    For bSortIndex = (cSamplesToTake - 1) To 0 Step -1  ' Create a loop for the amount of samples to take
        wSampleArray[bSortIndex] = ReadADC(ADC_CH1)     ' Read the channel into the filter array
        DelayUS 2                                       ' A small delay to allow the ADC's capacitor to re-charge
    Next                                                ' Close the loop
    GoSub MedianFilter                                  ' Perform a median filter on the samples
    wYoutput = wSortResult                              ' Transfer the result of the filter
    fYVoltage = wYoutput * cQuanta                      ' Convert to a voltage
    Return
'--------------------------------------------------------------------
' Read the Z output from the accelerometer (Up and Down)
'
' Input     : None
' Output    : wZoutput holds the filtered 10-bit result from the ADC
'             fZVoltage holds the voltage output from the channel
' Notes     : None
ReadChannel_Z:
    For bSortIndex = (cSamplesToTake - 1) To 0 Step -1  ' Create a loop for the amount of samples to take
        wSampleArray[bSortIndex] = ReadADC(ADC_CH2)     ' Read the channel into the filter array
        DelayUS 2                                       ' A small delay to allow the ADC's capacitor to re-charge
    Next                                                ' Close the loop
    GoSub MedianFilter                                  ' Perform a median filter on the samples
    wZoutput = wSortResult                              ' Transfer the result of the filter
    fZVoltage = wZoutput * cQuanta                      ' Convert to a voltage
    Return
'--------------------------------------------------------------------
' The main program loop starts here
Main:
    Set_2g()                                          ' Set accelerometer sensitivity
'
' Open the ADC and make AN0, AN1, and AN2 analogue inputs
'
    OpenADC(ADC_FOSC_32 & ADC_RIGHT_JUST & ADC_0_TAD, ADC_REF_VDD_VSS, ADC_1ANA & ADC_2ANA & ADC_3ANA)
'
' Display the accelerometer values on the serial terminal
'
    While 1 = 1                             ' Create an infinite loop
        GoSub ReadChannel_X                 ' Read X channel
        GoSub ReadChannel_Y                 ' Read Y channel
        GoSub ReadChannel_Z                 ' Read Z channel
        '
        ' Calculate the gravity measurement of the axis
        '
        fGravity_X = (fXVoltage - Zero_G_X) / cSensitivity
        fGravity_Y = (fYVoltage - Zero_G_Y) / cSensitivity
        fGravity_Z = (fZVoltage - Zero_G_Z) / cSensitivity
        '
        ' Convert the gravity measurements to angles
        '
        Rx = fGravity_X * fGravity_X
        Ry = fGravity_Y * fGravity_Y
        Rz = fGravity_Z * fGravity_Z

        R = Sqr(Rx + Ry + Rz)
        '
        ' Convert to integer degrees
        '
        bAngleX = (ACos(fGravity_X / R)) * cDegreeConversion
        bAngleY = (ACos(fGravity_Y / R)) * cDegreeConversion

        HRSOut 1                            ' Clear the serial terminal's window
        HRSOut "X axis (Forward-Backward): ", Dec2 fGravity_X, " g\r"     ' Display the X channel's gravity value
        HRSOut "Y axis (Side to Side)    : ", Dec2 fGravity_Y, " g\r"     ' Display the Y channel's gravity value
        HRSOut "\r"
        HRSOut "X axis (Forward-Backward): ", Dec bAngleX, " degrees\r"   ' Display the X channel's angle
        HRSOut "Y axis (Side to Side)    : ", Dec bAngleY, " degrees\r"   ' Display the Y channel's angle

        DelayMS 300
    Wend                                    ' Do it forever
